(*
FPU i87 implementation of vector operations
(C) Copyright by Jan Jelowicki, jasj@karnet.ar.wroc.pl

This implements basic level math operations

This is free source code. Please let me know about bugs.
Please do not remove my name from this file.
No guarantee is given etc...
*)
{$i optimise.inc}
unit scalars;
interface

uses
  use32;

type { base types }
  sfloat = single;
  psfloat = ^sfloat;
  dfloat = double;
  pdfloat = ^dfloat;
  qfloat = extended;
  pqfloat = ^qfloat;

type { semantic types }
  float = dfloat;
  pfloat = pdfloat;
  vfloat = qfloat;
  pvfloat = pqfloat;

type
  pbool = ^boolean;
  pbyte = ^byte;
  index = word;
  pindex = ^index;
  pword = ^word;
  pinteger = ^integer;

const
  sflsize = sizeof(sfloat);
  dflsize = sizeof(dfloat);
  qflsize = sizeof(qfloat);
  idxsize = sizeof(index);
  ptrsize = sizeof(pointer);
   flsize = sizeof(float);

type
  realfunction = function(x: vfloat; param: pointer): vfloat;
  realfunction2 = function(x,y: vfloat; param: pointer): vfloat;

  { returns fnc result and 1st derivative }
  realfunctiond = function(x: vfloat; p: pointer; var df: float): vfloat;

const
  { Cartesian axes }
  xxx = 0;
  yyy = 1;
  zzz = 2;
  ttt = 3;
  { directions }
  minus = xxx;
  plus = yyy;

const
  not0 = 1e-10;
  macheps: vfloat = 0.0; { dynamic machine zero }

  function sgndigits(x: vfloat): integer;

var
  sqrt2,sqrt3,ln2,ln10,eee: vfloat;

  { basic support for float data type }

  function pow(x,y: vfloat): vfloat;
  { fast power with integer exponent, any base }
  function ipow(X: Extended; I: Integer): Extended;
  function iipow(X: Integer; I: Integer): Longint;

  { @ <-- m(a,b) }
  function flmax(const x,y: float): float;
  function flmin(const x,y: float): float;
  procedure flmaxmin(const x,y: float; var max,min: float);
  function imax(const x,y: longint): longint;
  function imin(const x,y: longint): longint;
  procedure imaxmin(const x,y: integer; var max,min: integer);

  { Smallest integer >= X, |X| < MaxInt }
  function Ceil(X: Extended): Longint;
  { Largest integer <= X,  |X| < MaxInt }
  function Floor(X: Extended): Longint;

  function sign(const x: vfloat): shortint;
  {$ifdef virtualpascal}
  inline;
  begin
    if (x>macheps) then sign := 1
    else if (x<-macheps) then sign := -1
    else sign := 0;
  end; {sign}
  {$endif}

  function msign(const dir: byte): shortint;             {PLUS/MINUS --> +1/-1}
  {$ifdef virtualpascal}
  inline;
  begin
    msign := (dir shl 1) - 1;
  end; {msign}
  {$endif}

  function smoothsign(const x: vfloat; p: pointer): vfloat; far;
  function smoothsignd(const x: vfloat; p: pointer; var dx: float): vfloat; far;

  function positive(const x: vfloat): vfloat;

  { @ <-- |(x,y)| }
  function v2absfl(const x,y: float): vfloat;
  function v3absfl(const x,y,z: float): vfloat;

  procedure sflswap(var x,y: sfloat);
  procedure  flswap(var x,y: dfloat);
  procedure qflswap(var x,y: qfloat);
  procedure idxswap(var x,y: index);
  procedure intswap(var x,y: integer);


implementation

{$ifdef asm_math} {$i scalars.asm} {$else} {$i scalars.pp} {$endif}

(***

  { fast power with integer exponent, any base }

  function ipow0(const x: vfloat; n: integer): vfloat;
  begin
    if (n>0) then begin
      if odd(n) then
        ipow0 := x*ipow0(x,n-1)
      else
        ipow0 := sqr(ipow0(x,n div 2));
    end else if (n=0) then
      ipow0 := 1
    else if (x<>0) then
      ipow0 := 1/ipow0(x,-n)
    else
      ipow0 := 0;
  end;

  function ipow1(x: vfloat; n: integer): vfloat;
  var
    w,v: vfloat;
    i: integer;
  begin
    if (n>0) then begin
      w := 1.0;
      v := x;
      i := n;
      while (i>0) do begin
        if odd(i) then
          w := w*v;
        i := i div 2;
        v := sqr(v);
      end;
      ipow1 := w;
    end else if (n=0) then
      ipow0 := 1
    else if (x<>0) then
      ipow0 := 1/ipow0(x,-n)
    else
      ipow0 := 0;
  end;

  function ipow2;
  var
    i: longint;
    r: vfloat;
  begin
    if (n>0) then begin
      i := 1;
      while (i<n) do
        i := i shl 1; {2*i}
      r := 1;
      while (i>0) do begin
        r := sqr(r);
        if ((i and n) >0) then
          r := x*r;
        i := i shr 1; {i div 2}
      end;
      ipow2 := r;
    end else if (n=0) then
      ipow2 := 1
    else if (x<>0) then
      ipow2 := 1/ipow2(x,-n)
    else
      ipow2 := 0;
  end;

(***)

  { Invariant: Y >= 0 & Result*X**Y = X**I.  Init Y = I and Result = 1. }
  function iipow(X: Integer; I: Integer): Longint;
  var
    Y: Integer;
    P: Longint;
    XX: Longint;
  begin
    Y := Abs(I);
    P := 1;
    XX := X;
    while (Y>0) do begin
      while not Odd(Y) do
      begin
        Y := Y shr 1;
        XX := XX * XX;
      end;
      Dec(Y);
      P := P * XX;
    end;
    if (i>=0) then
      iipow := p
    else
      iipow := 0;
  end;

  function smoothsign;
  var
    xe,eps: vfloat;
  begin
    eps := pfloat(p)^;
    if (x<-eps) then
      smoothsign := -1.0
    else if (x<eps) then begin
      xe := x/eps;
      smoothsign := 0.5*xe*(3-sqr(xe));
    end else
      smoothsign := 1.0;
  end; {sign}

  function smoothsignd;
  var
    xe,xe2,eps: vfloat;
  begin
    eps := pfloat(p)^;
    dx := 0.0;
    if (x<-eps) then
      smoothsignd := -1.0
    else if (x<eps) then begin
      xe := x/eps;
      xe2 := sqr(xe);
      smoothsignd := 0.5*xe*(3-xe2);
      dx := 3/(2*eps)*(1-xe2);
    end else
      smoothsignd := 1.0;
  end; {sign}

  function Ceil;
  var
    r: longint;
  begin
    R := Trunc(X);
    if (Frac(X)>0) then
      Inc(R);
    ceil := r;
  end;

  function Floor;
  var
    r: longint;
  begin
    R := Trunc(X);
    if (Frac(X)<0) then
      Dec(R);
    floor := r;
  end;

  function detectmachinezero: vfloat;
  var
    r: vfloat;
  begin
    r := 1.0;
    while (1.0+r<>1.0) do
      r := 0.5*r;
    detectmachinezero := r;
  end; {detectmachinezero}

  function sgndigits;
  var
    i: integer;
  begin
    if (abs(x)>macheps) then begin
      i := round(ln(abs(x))/ln10);
      if (i>0) then sgndigits := 0
      else sgndigits := -i+1;
    end else
      sgndigits := 1;
  end;

  {$ifndef virtualpascal}
  function sign;
  begin
    if (abs(x)<macheps) then sign := 0
    else if (x>0) then sign := 1
    else sign := -1;
  end; {sign}

  function msign;
  begin
    msign := (dir shl 1) - 1;
  end; {msign}
  {$endif}

begin
  macheps := detectmachinezero;
  ln2 := ln(2);
  ln10 := ln(10);
  sqrt2 := sqrt(2);
  sqrt3 := sqrt(3);
  eee := exp(1);
end.


