{
  BLAS module, dense and sparse operations.

  i387-asm optimised code for Single, Double nad Extended precision.
  Results of ASM implementations may differ to the standard ones
  due to reversed summation direction.

  Tested with
    fPrint Virtual Pascal 2.0 (asm and pascal codes)
    Borland Pascal 7.0 and FPC (pascal code)

  Copyright (c) 1996-2000 by Jan Jeowicki,
                Department of Mathematics, Wrocaw University of Agriculture, Poland
                jasj@karnet.ar.wroc.pl
                www.ar.wroc.pl/~jasj

  Partially inspired with VMath10 by Wolfgang Lieff (c) 1991
                          Flinders Institute for Atmospheric and Marine Sciences
                          Bedford Park , South Australia 5042
                          mowl@cc.flinders.edu.au

This is free source code. Please let me know about bugs.
Please do not remove my name from this file.
No guarantee is given etc...
}

{$i optimise.inc}
unit vectors;
interface
uses
  mops,scalars,use32;

const
  vecmaxdim = maxblock div sizeof(dFloat);

type { floating-point vector base types }

  sVector = array[1..vecmaxdim] of sFloat;
  psVector = ^sVector;

  dVector = array[1..vecmaxdim] of dFloat;
  pdVector = ^dVector;

  qVector = array[1..vecmaxdim div 2] of qFloat;
  pqVector = ^qVector;

  { base precision is double precision }
  Vector = dVector;
  pVector = pdVector;

  pfVector = array[1..vecmaxdim] of pdFloat;
  ppfVector = ^pfVector;

  pvVector = array[1..vecmaxdim] of pVector;
  ppvVector = ^pvVector;

  { index vector }
  iVector = array[1..vecmaxdim] of index;
  piVector = ^iVector;

  { automatic index vector }
  aivector = record
               valid: boolean;
               allocdimen: index;
               dimen: index;
               ivdata: ivector;
             end;
  paivector = ^aivector;

{ conversion utilities }

  { Dense <--> Sparse data storage }
  { spv <-- v }
  procedure veccompress(const dim: index; var pattern: ivector; var v, cv: vector);
  { v <-- spv }
  procedure vecuncompress(const dim: index; var cv: vector; var pattern: ivector; var v: vector);

  { Y <-- X with data representation conversion }
  procedure dVec2qVec(const dim0,dim1: index; var x: dVector; var y: qVector);
  procedure qVec2dVec(const dim0,dim1: index; var x: qVector; var y: dVector);
  procedure dVec2sVec(const dim0,dim1: index; var x: dVector; var y: sVector);
  procedure sVec2dVec(const dim0,dim1: index; var x: sVector; var y: dVector);

{ norms and comparing functions }

type
  { @ <-- |X| }
  dVecnormf = function(const dim0,dim1: index; var r: dVector): qFloat;
  spdVecnormf = function(const dim: index; var pattern: ivector; var r: dVector): qFloat;
  { @ <-- |X-Y| }
  dVecdistf = function(const dim0,dim1: index; var x,y: dVector): qFloat;
  { @ <-- (|X-Y|<Threshold) }
  dVeccmpf = function(const dim0,dim1: index; var x,y: dVector; const threshold: qFloat): boolean;
  spdVeccmpf = function(const dim: index; var pattern: ivector; var x,y: dVector; const threshold: qFloat): boolean;

  vecnormf = dVecnormf;
  veccmpf = dVecCmpF;

type
  { @ <-- |X| }
  qvecnormf = function(const dim0,dim1: index; var r: qVector): qFloat;
  spqvecnormf = function(const dim: index; var pattern: ivector; var r: qVector): qFloat;
  { @ <-- |X-Y| }
  qvecdistf = function(const dim0,dim1: index; var x,y: qVector): qFloat;
  { @ <-- (|X-Y|<Threshold) }
  qveccmpf = function(const dim0,dim1: index; var x,y: qVector; const threshold: qFloat): boolean;
  spqveccmpf = function(const dim: index; var pattern: ivector; var x,y: qVector; const threshold: qFloat): boolean;

const { norm exponent for L^p norm }
  VecLpNormExponent: dFloat = 2.0;

const { magic codes for vector norms }
  exactveccmp = 0;
  l1inorm  = 1;
  l2inorm  = 2;
  supinorm = 3;
  lpinorm  = 4;
  maxinorm = 4;

var
  vecnorm: array[1..maxinorm] of dVecnormf;
  spvecnorm: array[1..maxinorm] of spdVecnormf;
  vecdist: array[1..maxinorm] of dVecdistf;
  veccmp : array[0..maxinorm] of dVeccmpf;
  spveccmp : array[0..maxinorm] of spdVeccmpf;

var
  qvecnorm: array[1..maxinorm] of qvecnormf;
  spqvecnorm: array[1..maxinorm] of spqvecnormf;
  qvecdist: array[1..maxinorm] of qvecdistf;
  qveccmp : array[0..maxinorm] of qveccmpf;
  spqveccmp : array[0..maxinorm] of spqveccmpf;

(* vector operations *)

  { @ = |R| }
  function vecnorml1(const dim0,dim1: index; var r: dVector): qFloat; far;
  function vecnorml2(const dim0,dim1: index; var r: dVector): qFloat; far;
  function vecnormlp(const dim0,dim1: index; var r: dVector): qFloat; far;
  function vecnormlsup(const dim0,dim1: index; var r: dVector): qFloat; far;

  function spvecnorml1(const dim: index; var pattern: ivector; var r: dVector): qFloat; far;
  function spvecnorml2(const dim: index; var pattern: ivector; var r: dVector): qFloat; far;
  function spvecnormlp(const dim: index; var pattern: ivector; var r: dVector): qFloat; far;
  function spvecnormlsup(const dim: index; var pattern: ivector; var r: dVector): qFloat; far;

  function qvecnorml1(const dim0,dim1: index; var r: qVector): qFloat; far;
  function qvecnorml2(const dim0,dim1: index; var r: qVector): qFloat; far;
  function qvecnormlp(const dim0,dim1: index; var r: qVector): qFloat; far;
  function qvecnormlsup(const dim0,dim1: index; var r: qVector): qFloat; far;

  function spqvecnorml1(const dim: index; var pattern: ivector; var r: qVector): qFloat; far;
  function spqvecnorml2(const dim: index; var pattern: ivector; var r: qVector): qFloat; far;
  function spqvecnormlp(const dim: index; var pattern: ivector; var r: qVector): qFloat; far;
  function spqvecnormlsup(const dim: index; var pattern: ivector; var r: qVector): qFloat; far;

  { @ = |X-Y| }
  function vecdistl1(const dim0,dim1: index; var x,y: dVector): qFloat; far;
  function vecdistl2(const dim0,dim1: index; var x,y: dVector): qFloat; far;
  function vecdistlp(const dim0,dim1: index; var x,y: dVector): qFloat; far;
  function vecdistlsup(const dim0,dim1: index; var x,y: dVector): qFloat; far;

  function qvecdistl1(const dim0,dim1: index; var x,y: qVector): qFloat; far;
  function qvecdistl2(const dim0,dim1: index; var x,y: qVector): qFloat; far;
  function qvecdistlp(const dim0,dim1: index; var x,y: qVector): qFloat; far;
  function qvecdistlsup(const dim0,dim1: index; var x,y: qVector): qFloat; far;

  { @ = (Y==X), exact logical comparison }
  function veccmp0(const dim0,dim1: index; var x,y: dVector; const dummy: qFloat): boolean; far;
  { @ = (Y==X) up to Threshold with INorm }
  function veccmpl1(const dim0,dim1: index; var x,y: dVector; const threshold: qFloat): boolean; far;
  function veccmpl2(const dim0,dim1: index; var x,y: dVector; const threshold: qFloat): boolean; far;
  function veccmplp(const dim0,dim1: index; var x,y: dVector; const threshold: qFloat): boolean; far;
  function veccmplsup(const dim0,dim1: index; var x,y: dVector; const threshold: qFloat): boolean; far;

  { @ = (Y==X), exact logical comparison }
  function qveccmp0(const dim0,dim1: index; var x,y: qVector; const dummy: qFloat): boolean; far;
  { @ = (Y==X) up to Threshold with INorm }
  function qveccmpl1(const dim0,dim1: index; var x,y: qVector; const threshold: qFloat): boolean; far;
  function qveccmpl2(const dim0,dim1: index; var x,y: qVector; const threshold: qFloat): boolean; far;
  function qveccmplp(const dim0,dim1: index; var x,y: qVector; const threshold: qFloat): boolean; far;
  function qveccmplsup(const dim0,dim1: index; var x,y: qVector; const threshold: qFloat): boolean; far;

  { compare sparse vectors }
  function spveccmp0(const dim: index; var pattern: ivector; var x,y: dVector; const threshold: qFloat): boolean; far;
  function spveccmpl1(const dim: index; var pattern: ivector; var x,y: dVector; const threshold: qFloat): boolean; far;
  function spveccmpl2(const dim: index; var pattern: ivector; var x,y: dVector; const threshold: qFloat): boolean; far;
  function spveccmplp(const dim: index; var pattern: ivector; var x,y: dVector; const threshold: qFloat): boolean; far;
  function spveccmplsup(const dim: index; var pattern: ivector; var x,y: dVector; const threshold: qFloat): boolean; far;

  function spqveccmp0(const dim: index; var pattern: ivector; var x,y: qVector; const threshold: qFloat): boolean; far;
  function spqveccmpl1(const dim: index; var pattern: ivector; var x,y: qVector; const threshold: qFloat): boolean; far;
  function spqveccmpl2(const dim: index; var pattern: ivector; var x,y: qVector; const threshold: qFloat): boolean; far;
  function spqveccmplp(const dim: index; var pattern: ivector; var x,y: qVector; const threshold: qFloat): boolean; far;
  function spqveccmplsup(const dim: index; var pattern: ivector; var x,y: qVector; const threshold: qFloat): boolean; far;

  { R = X/|X|}
  function vecnormalize(const dim0,dim1: index; var x,r: dVector): qFloat;
  function qvecnormalize(const dim0,dim1: index; var x,r: qVector): qFloat;

  { X = 0 }
  procedure veczero(const dim0,dim1: index; var x: dVector);
  procedure qveczero(const dim0,dim1: index; var x: qVector);
  procedure sveczero(const dim0,dim1: index; var x: sVector);
  procedure spveczero(const dim: index; var pattern: ivector; var x: dVector);
  procedure spqveczero(const dim: index; var pattern: ivector; var x: qVector);

  { R = E_i }
  procedure vecei(const dim, i: index; var r: dVector);
  procedure qvecei(const dim, i: index; var r: qVector);

  { R = [c] }
  procedure vecfill(const dim0,dim1: index; var r: dVector; const c: qFloat);
  procedure qvecfill(const dim0,dim1: index; var r: qVector; const c: qFloat);
  procedure spvecfill(const dim: index; var pattern: ivector; var r: dVector; const c: qFloat);
  procedure spqvecfill(const dim: index; var pattern: ivector; var r: qVector; const c: qFloat);

  { R[i] = c0 + (i-1) c }
  procedure vecenum(const dim0,dim1: index; var r: dVector; const c0,c: qFloat);
  procedure qvecenum(const dim0,dim1: index; var r: qVector; const c0,c: qFloat);
  procedure spvecenum(const dim: index; var pattern: ivector; var r: dVector; const c0,c: qFloat);
  procedure spqvecenum(const dim: index; var pattern: ivector; var r: qVector; const c0,c: qFloat);

  { @ = |R| }
  function vecsum(const dim0,dim1: index; var r: dVector): qFloat;
  function spvecsum(const dim: index; var pattern: ivector; var r: dVector): qFloat;
  function qvecsum(const dim0,dim1: index; var r: qVector): qFloat;
  function spqvecsum(const dim: index; var pattern: ivector; var r: qVector): qFloat;

  { @ = P.Q }
  function vecdot(const dim0,dim1: index; var p,q: dVector): qFloat;
  function spvecdot(const dim: index; var pattern: ivector; var p,q: dVector): qFloat;
  function qvecdot(const dim0,dim1: index; var p,q: qVector): qFloat;
  function spqvecdot(const dim: index; var pattern: ivector; var p,q: qVector): qFloat;

  { @ = P.P }
  function vecsqr(const dim0,dim1: index; var p: dVector): qFloat;
  function spvecsqr(const dim: index; var pattern: ivector; var p: dVector): qFloat;
  function qvecsqr(const dim0,dim1: index; var p: qVector): qFloat;
  function spqvecsqr(const dim: index; var pattern: ivector; var p: qVector): qFloat;

  { R = P*Q pointwise }
  procedure vectimes(const dim0,dim1: index; var p,q, r: dVector);
  procedure spvectimes(const dim: index; var pattern: ivector; var p,q, r: dVector);
  procedure qvectimes(const dim0,dim1: index; var p,q, r: qVector);
  procedure spqvectimes(const dim: index; var pattern: ivector; var p,q, r: qVector);

  { Y = X }
  procedure veccopy(const dim0,dim1: index; var x,y: dVector);
  procedure spveccopy(const dim: index; var pattern: ivector; var x,y: dVector);
  procedure qveccopy(const dim0,dim1: index; var x,y: qVector);
  procedure spqveccopy(const dim: index; var pattern: ivector; var x,y: qVector);

  { Y = -X }
  procedure vecmcopy(const dim0,dim1: index; var x,y: dVector);
  procedure spvecmcopy(const dim: index; var pattern: ivector; var x,y: dVector);
  procedure qvecmcopy(const dim0,dim1: index; var x,y: qVector);
  procedure spqvecmcopy(const dim: index; var pattern: ivector; var x,y: qVector);

  { R = X + a }
  procedure vecaddc(const dim0,dim1: index; var x: dVector; const a: qFloat; var r: dVector);
  procedure spvecaddc(const dim: index; var pattern: ivector; var x: dVector; const a: qFloat; var r: dVector);
  procedure qvecaddc(const dim0,dim1: index; var x: qVector; const a: qFloat; var r: qVector);
  procedure spqvecaddc(const dim: index; var pattern: ivector; var x: qVector; const a: qFloat; var r: qVector);

  { R = a X }
  procedure vecscale(const dim0,dim1: index; var x: dVector; const a: qFloat; var r: dVector);
  procedure spvecscale(const dim: index; var pattern: ivector; var x: dVector; const a: qFloat; var r: dVector);
  procedure qvecscale(const dim0,dim1: index; var x: qVector; const a: qFloat; var r: qVector);
  procedure spqvecscale(const dim: index; var pattern: ivector; var x: qVector; const a: qFloat; var r: qVector);

  { R = X + Y }
  procedure vecadd(const dim0,dim1: index; var x,y: dVector; var r: dVector);
  procedure spvecadd(const dim: index; var pattern: ivector; var x,y: dVector; var r: dVector);
  procedure qvecadd(const dim0,dim1: index; var x,y: qVector; var r: qVector);
  procedure spqvecadd(const dim: index; var pattern: ivector; var x,y: qVector; var r: qVector);

  { R = X - Y }
  procedure vecsub(const dim0,dim1: index; var x,y: dVector; var r: dVector);
  procedure spvecsub(const dim: index; var pattern: ivector; var x,y: dVector; var r: dVector);
  procedure qvecsub(const dim0,dim1: index; var x,y: qVector; var r: qVector);
  procedure spqvecsub(const dim: index; var pattern: ivector; var x,y: qVector; var r: qVector);

  { R = p(X + Y) }
  procedure vecpadd(const dim0,dim1: index; const p: qFloat; var x,y: dVector; var r: dVector);
  procedure spvecpadd(const dim: index; var pattern: ivector; const p: qFloat; var x,y: dVector; var r: dVector);
  procedure qvecpadd(const dim0,dim1: index; const p: qFloat; var x,y: qVector; var r: qVector);
  procedure spqvecpadd(const dim: index; var pattern: ivector; const p: qFloat; var x,y: qVector; var r: qVector);

  { R = p(X - Y) }
  procedure vecpsub(const dim0,dim1: index; const p: qFloat; var x,y: dVector; var r: dVector);
  procedure spvecpsub(const dim: index; var pattern: ivector; const p: qFloat; var x,y: dVector; var r: dVector);
  procedure qvecpsub(const dim0,dim1: index; const p: qFloat; var x,y: qVector; var r: qVector);
  procedure spqvecpsub(const dim: index; var pattern: ivector; const p: qFloat; var x,y: qVector; var r: qVector);

  { R = p X + Y }
  procedure vecaddpxy(const dim0,dim1: index; const p: qFloat; var x: dVector; var y: dVector; var r: dVector);
  procedure spvecaddpxy(const dim: index; var pattern: ivector; const p: qFloat;
                        var x: dVector; var y: dVector; var r: dVector);
  procedure qvecaddpxy(const dim0,dim1: index; const p: qFloat; var x: qVector; var y: qVector; var r: qVector);
  procedure spqvecaddpxy(const dim: index; var pattern: ivector; const p: qFloat;
                        var x: qVector; var y: qVector; var r: qVector);

  { R = p X - Y }
  procedure vecsubpxy(const dim0,dim1: index; const p: qFloat; var x: dVector; var y: dVector; var r: dVector);
  procedure spvecsubpxy(const dim: index; var pattern: ivector;
                        const p: qFloat; var x: dVector; var y: dVector; var r: dVector);
  procedure qvecsubpxy(const dim0,dim1: index; const p: qFloat; var x: qVector; var y: qVector; var r: qVector);
  procedure spqvecsubpxy(const dim: index; var pattern: ivector;
                        const p: qFloat; var x: qVector; var y: qVector; var r: qVector);

  { R = p X + q Y }
  procedure vecaddpxqy(const dim0,dim1: index;
                       const p: qFloat; var x: dVector; const q: qFloat; var y: dVector; var r: dVector);
  procedure spvecaddpxqy(const dim: index; var pattern: ivector; const p: qFloat; var x: dVector;
                                                                 const q: qFloat; var y: dVector; var r: dVector);
  procedure qvecaddpxqy(const dim0,dim1: index;
                       const p: qFloat; var x: qVector; const q: qFloat; var y: qVector; var r: qVector);
  procedure spqvecaddpxqy(const dim: index; var pattern: ivector; const p: qFloat; var x: qVector;
                                                                 const q: qFloat; var y: qVector; var r: qVector);

  { Y = f(X) componentwise }
  procedure vectransform(const dim0,dim1: index; var x: dVector; f: realfunction; p: pointer; var y: dVector);
  procedure spvectransform(const dim: index; var pattern: ivector;
                           var x: dVector; f: realfunction; p: pointer; var y: dVector);
  procedure qvectransform(const dim0,dim1: index; var x: qVector; f: realfunction; p: pointer; var y: qVector);
  procedure spqvectransform(const dim: index; var pattern: ivector;
                           var x: qVector; f: realfunction; p: pointer; var y: qVector);

  { @ = max/min(R_i) }
  function vecmin(const dim0,dim1: index; var r: dVector): dFloat;
  function spvecmin(const dim: index; var pattern: ivector; var r: dVector): dFloat;
  function qvecmin(const dim0,dim1: index; var r: qVector): qFloat;
  function spqvecmin(const dim: index; var pattern: ivector; var r: qVector): qFloat;

  function vecmax(const dim0,dim1: index; var r: dVector): dFloat;
  function spvecmax(const dim: index; var pattern: ivector; var r: dVector): dFloat;
  function qvecmax(const dim0,dim1: index; var r: qVector): qFloat;
  function spqvecmax(const dim: index; var pattern: ivector; var r: qVector): qFloat;

  procedure vecminmax(const dim0,dim1: index; var r: dVector; var min,max: dFloat);
  procedure spvecminmax(const dim: index; var pattern: ivector; var r: dVector; var min,max: dFloat);
  procedure qvecminmax(const dim0,dim1: index; var r: qVector; var min,max: qFloat);
  procedure spqvecminmax(const dim: index; var pattern: ivector; var r: qVector; var min,max: qFloat);

  function ivecmin(const dim0,dim1: index; var r: ivector): index;
  function ivecmax(const dim0,dim1: index; var r: ivector): index;
  procedure ivecminmax(const dim0,dim1: index; var r: ivector; var min,max: index);

  { @ = i: (R_i=max/min(R_j)) }
  function vecimin(const dim0,dim1: index; var r: dVector): index;
  function vecimax(const dim0,dim1: index; var r: dVector): index;
  function qvecimin(const dim0,dim1: index; var r: qVector): index;
  function qvecimax(const dim0,dim1: index; var r: qVector): index;

  function ivecimin(const dim0,dim1: index; var r: ivector): index;
  function ivecimax(const dim0,dim1: index; var r: ivector): index;

  { @ = i: (R_i=max/min(abs(R_j))) }
  function veciamin(const dim0,dim1: index; var r: dVector): index;
  function veciamax(const dim0,dim1: index; var r: dVector): index;
  function qveciamin(const dim0,dim1: index; var r: qVector): index;
  function qveciamax(const dim0,dim1: index; var r: qVector): index;

  function iveciamin(const dim0,dim1: index; var r: ivector): index;
  function iveciamax(const dim0,dim1: index; var r: ivector): index;

(* cyclic shift C cells right *)
  procedure vecshift(const dim0,dim1: index; var x: dvector; const c: integer; var y: dvector);
  procedure spvecshift(const dim: index; var pattern: ivector; var x: dvector; const c: integer; var y: dvector);
  procedure ivecshift(const dim0,dim1: index; var x: ivector; const c: integer; var y: ivector);
  procedure svecshift(const dim0,dim1: index; var x: svector; const c: integer; var y: svector);
  procedure qvecshift(const dim0,dim1: index; var x: qvector; const c: integer; var y: qvector);

(* invert element ordering *)
  procedure vecreflect(const dim0,dim1: index; var x: dvector; var y: dvector);
  procedure ivecreflect(const dim0,dim1: index; var x: ivector; var y: ivector);
  procedure svecreflect(const dim0,dim1: index; var x: svector; var y: svector);
  procedure qvecreflect(const dim0,dim1: index; var x: qvector; var y: qvector);

(* sorting/indexing algorithms *)

  { QuickSort }
  procedure vecqsort(il,ir:index; var a: dVector);
  procedure qvecqsort(il,ir:index; var a: qVector);
  { HeapSort }
  procedure vechsort(il,ir:index; var a: dVector);
  procedure qvechsort(il,ir:index; var a: qVector);

(* memory allocation service *)

  { static vectors }
  function vecalloc(const dim0,dim1: index; var p: pdVector): boolean;
  procedure vecfree(const dim0,dim1: index; var p: pdVector);
  function vecrealloc(const dim0,olddim1,newdim1: index; var p: pdVector): boolean;

  function qvecalloc(const dim0,dim1: index; var p: pqVector): boolean;
  procedure qvecfree(const dim0,dim1: index; var p: pqVector);
  function qvecrealloc(const dim0,olddim1,newdim1: index; var p: pqVector): boolean;

  function svecalloc(const dim0,dim1: index; var p: psVector): boolean;
  procedure svecfree(const dim0,dim1: index; var p: psVector);
  function svecrealloc(const dim0,olddim1,newdim1: index; var p: psVector): boolean;

  { swap vector pointers }
  procedure svecswap(var x,y: psVector);
  procedure vecswap(var x,y: pdVector);
  procedure qvecswap(var x,y: pqVector);

  { interface to text stream }
  procedure vecread(const dim0,dim1: index; var v: dVector; var f: text; msg: string; const col: index);
  procedure vecwrite(const dim0,dim1: index; var v: dVector; var f: text; msg: string; const col: index);
  procedure spvecread(const dim: index; var pattern: ivector; var v: dVector; var f: text; msg: string; const col: index);
  procedure spvecwrite(const dim: index; var pattern: ivector; var v: dVector; var f: text; msg: string; const col: index);
  procedure qvecread(const dim0,dim1: index; var v: qVector; var f: text; msg: string; const col: index);
  procedure qvecwrite(const dim0,dim1: index; var v: qVector; var f: text; msg: string; const col: index);
  procedure spqvecread(const dim: index; var pattern: ivector; var v: qVector; var f: text; msg: string; const col: index);
  procedure spqvecwrite(const dim: index; var pattern: ivector; var v: qVector; var f: text; msg: string; const col: index);

  function pfvecalloc(const dim0,dim1: integer; var p: ppfvector): boolean;
  procedure pfvecfree(const dim0,dim1: integer; var p: ppfvector);
  procedure pfveczero(const dim0,dim1: integer; var p: pfvector);

(* general workspace definition and support *)

const
  maxvecwspace = 10;
type
  pvecwspacetype = ^vecwspacetype;
  vecwspacetype = record
                    nwspace: index;
                    wssize: pivector;
                    wspace: array[1..maxvecwspace] of pdVector;
                  end;

  function vecwspacealloc(nvar: index; var wspace: vecwspacetype; n: index): boolean;
  procedure vecwspacefree(var wspace: vecwspacetype);

{
    Gaussian elimination for multidiagonal matrices

     |c d e z . . . . . . .|   |y|   |f|
     |b c d e z . . . . . .|   |y|   |f|
     |a b c d e z . . . . .|   |y|   |f|
     |x a b c d e z . . . .|   |y|   |f|
     |. x a b c d e z . . .|   |y|   |f|
     |. . x a b c d e z . .| * |y| = |f|
     |. . . x a b c d e z .|   |y|   |f|
     |. . . . x a b c d e z|   |y|   |f|
     |. . . . . x a b c d e|   |y|   |f|
     |. . . . . . x a b c d|   |y|   |f|
     |. . . . . . . x a b c|   |y|   |f|

    with possible partial pivoting in rows. Sources:
    Samarskij, Nikolayev, "Metody rozwizywania rwna siatkowych", Moscow "Nauka" 1978,
    Press et al. "Numerical Recipes in C", Cambridge Univ. Press 1994
}

  { Diag[acb] y = f , monotone sweep }
  function diag3m(dim: index;
                  var a {subdiagonal}, c {main diagonal}, b {superdiagonal},
                      f {residuals}: dVector;
                  var y {solution}: dVector): boolean;

  { Diag[acb] y = f  with row pivoting }
  function diag3(dim: index;
                 var a {subdiagonal}, c {main diagonal}, b {superdiagonal},
                     f {residuals}: dVector;
                 var y {solution}: dVector): boolean;

  { Cyclic tridiagonal system
              |.....B|
    Diag[acb]+|      | y = f , with Sherman-Morrison formula
              |A.....|
    Bugs!!!
  }
  function diag3c(dim: index;
                  var a {subdiagonal}, c {main diagonal}, b {superdiagonal}: dVector;
                  const alpha {lower-left entry}, beta {upper-right entry}: dFloat;
                  var f {residuals}: dVector;
                  var y {solution}: dVector): boolean;

  { Diag[abcde] y = f with row pivoting }
  function diag5(dim: index;
                 var a,b,{subdiagonals 2 and 1}
                     c {main diagonal},
                     d,e,{superdiagonals 1 and 2}
                     f {residuals}: dVector;
                 var y {solution}: dVector): boolean;

  { Diag[xabcdez] y = f with row pivoting (bugs!?) }
  function diag7(dim: index;
                 var x,a,b,{subdiagonals 3,2 and 1}
                     c {main diagonal},
                     d,e,z,{superdiagonals 1,2 and 3}
                     f {residuals}: dVector;
                 var y {solution}: dVector): boolean;

  { IVector type support }

  function ivalloc(const dim0,dim1: index; var p: pivector): boolean;
  function ivrealloc(const dim0,olddim1,newdim1: index; var p: pivector): boolean;
  procedure ivfree(const dim0,dim1: index; var p: pivector);

  function aivalloc(dim: index; var p: paivector): boolean;
  function aivrealloc(dim1: index; var p: paivector): boolean;
  procedure aivfree(var p: paivector);

  procedure ivzero(const dim0,dim1: index; var p: ivector);
  procedure ivcopy(const dim0,dim1: index; var p,q: ivector);
  procedure ivfill(const dim0,dim1: index; var p: ivector; const c: index);
  procedure ivenum(const dim0,dim1: index; var v: ivector; const c0,c: index);

  procedure aivzero(var p: aivector);
  function aivcopy(var p,q: aivector): boolean;
  procedure aivfill(var p: aivector; const c: index);
  procedure aivenum(var a: aivector);

{ Structural operations are not supported for these types }
type
  pmtxdiag3  = ^mtxdiag3;
  mtxdiag3  = array[-1..1] of pdVector;

  pmtxdiag5  = ^mtxdiag5;
  mtxdiag5  = array[-2..2] of pdVector;

  pmtxdiag7  = ^mtxdiag7;
  mtxdiag7  = array[-3..3] of pdVector;

(* Searching utility *)
  {  Binary search, X _MUST_ be increasingly ordered
     n = vector length (_MUST_ START FROM 1 !!!)
     x = vector
     x0= value being searched
     i <-- the last index with x_i<=x0
     d <-- the ratio (x0-x[i])/(x[i+1]-x[i])
     @ <-- x[1] <= x0 <= x[n]
  }
  function vecsearch(n: index; var x: dVector; x0: qFloat; var i: index; var d: dFloat): boolean;
  function qvecsearch(n: index; var x: qVector; x0: qFloat; var i: index; var d: qFloat): boolean;

  { The same except providing internal slope estimates in the cases (i=0) and (i=n) }
  function vecsearchin(n: index; var x: dVector; x0: qFloat; var i: index; var d: dFloat): boolean;
  function qvecsearchin(n: index; var x: qVector; x0: qFloat; var i: index; var d: qFloat): boolean;

  { linear search }
  function vecsearchlin(const dim0,dim1: index; var x: dVector; x0: qFloat; var i: index): boolean;
  function vecsearchlinapprox(const dim0,dim1: index; var x: dVector; x0: qFloat; const tol: float; var i: index): boolean;

implementation

  {
    Here you shall find BLAS implementations for specified vector models:
  }

  {$i svectors}  { Single precision vectors }

  {$i dvectors}  { Double precision vectors }

  {$i qvectors}  { Extended precision vectors }

  {$i ivectors}  { Discrete vectors }

  {$i vecconv}   { Floating-point vector converters }

  {$i vecband}   { Gaussian algorithms for band-structured systems of linear eqns }

  {
    Remaining part of the body supports management of dVector lists
  }

  procedure vecwspacefree;
  var
    n: index;
  begin
    with wspace do begin
      n := nwspace;
      while (nwspace>0) do begin
        if (wssize^[nwspace]>0) then begin
          vecfree(1,wssize^[nwspace],wspace[nwspace]);
          wssize^[nwspace] := 0;
        end;
        dec(nwspace);
      end;
      ivfree(1,n,wssize);
    end;
  end; {vecwspacefree}

  function vecwspacealloc;
  var
    ok: boolean;
  begin
    vecwspacefree(wspace);
    ok := false;
    if (n<=maxvecwspace) then with wspace do begin
      ok := ivalloc(1,n,wssize);
      nwspace := n;
      n := 0;
      while ok and (n<nwspace) do begin
        if vecalloc(1,nvar,wspace[n+1]) then begin
          inc(n);
          wssize^[n] := nvar;
        end else
          ok := false;
      end;
    end;
    if not(ok) then
      vecwspacefree(wspace);
    vecwspacealloc := ok;
  end; {vecwspacealloc}

  function pfvecalloc;
  begin
    if memoryAlloc(p,(dim1-dim0+1)*sizeof(pdfloat)) then begin
      p := @p^[2-dim0];
      pfvecalloc := true;
    end else
      pfvecalloc := false;
  end;

  procedure pfvecfree;
  begin
    if assigned(p) then begin
      p := @p^[dim0];
      memoryDeAlloc(p,(dim1-dim0+1)*sizeof(pdfloat));
    end;
  end;

  procedure pfveczero;
  begin
    fillchar(p[dim0],(dim1-dim0+1)*sizeof(pdfloat),0);
  end;

begin

  vecnorm[l1inorm] := vecnorml1;
  vecnorm[l2inorm] := vecnorml2;
  vecnorm[supinorm]:= vecnormlsup;
  vecnorm[lpinorm] := vecnormlp;

  spvecnorm[l1inorm] := spvecnorml1;
  spvecnorm[l2inorm] := spvecnorml2;
  spvecnorm[supinorm]:= spvecnormlsup;
  spvecnorm[lpinorm] := spvecnormlp;

  vecdist[l1inorm] := vecdistl1;
  vecdist[l2inorm] := vecdistl2;
  vecdist[supinorm]:= vecdistlsup;
  vecdist[lpinorm] := vecdistlp;

  veccmp[exactveccmp] := veccmp0;
  veccmp[l1inorm] := veccmpl1;
  veccmp[l2inorm] := veccmpl2;
  veccmp[supinorm]:= veccmplsup;
  veccmp[lpinorm] := veccmplp;

  spveccmp[exactveccmp] := spveccmp0;
  spveccmp[l1inorm] := spveccmpl1;
  spveccmp[l2inorm] := spveccmpl2;
  spveccmp[supinorm]:= spveccmplsup;
  spveccmp[lpinorm] := spveccmplp;

  qvecnorm[l1inorm] := qvecnorml1;
  qvecnorm[l2inorm] := qvecnorml2;
  qvecnorm[supinorm]:= qvecnormlsup;
  qvecnorm[lpinorm] := qvecnormlp;

  spqVecnorm[l1inorm] := spqVecnorml1;
  spqVecnorm[l2inorm] := spqVecnorml2;
  spqVecnorm[supinorm]:= spqVecnormlsup;
  spqVecnorm[lpinorm] := spqVecnormlp;

  qvecdist[l1inorm] := qvecdistl1;
  qvecdist[l2inorm] := qvecdistl2;
  qvecdist[supinorm]:= qvecdistlsup;
  qvecdist[lpinorm] := qvecdistlp;

  qveccmp[exactveccmp] := qveccmp0;
  qveccmp[l1inorm] := qveccmpl1;
  qveccmp[l2inorm] := qveccmpl2;
  qveccmp[supinorm]:= qveccmplsup;
  qveccmp[lpinorm] := qveccmplp;

  spqVeccmp[exactveccmp] := spqVeccmp0;
  spqVeccmp[l1inorm] := spqVeccmpl1;
  spqVeccmp[l2inorm] := spqVeccmpl2;
  spqVeccmp[supinorm]:= spqVeccmplsup;
  spqVeccmp[lpinorm] := spqVeccmplp;

end.

